"""Represents an enemy"""

import math
import random

import serge.actor
import serge.blocks.actors

from theme import G
import common
import board

class MovementBlocked(Exception): """Cannot move to the desired goal"""


class Enemy(serge.blocks.actors.ScreenActor):
    """An enemy on the screen"""

    paused = False
    
    def __init__(self, tag, name, board, layout, parent, goal_index):
        """Initialise the Enemy"""
        super(Enemy, self).__init__(tag, name)
        self.setSpriteName(name)
        self.setLayerName('enemies')
        self.board = board
        self.layout = layout
        self.parent = parent
        self.goal_index = goal_index
        self.path = None
        self.goal = None
        self.speed = self.max_speed = G('speed', name)
        self.speed_recovery = G('speed-recovery', name)
        self.hp = G('hp', name)
        self.damage = G('damage', name)
        self.score = G('score', name)
        self.cash = G('cash', name)
        self.wait_limit = G('wait-limit', name)
        self.wait_restart = G('wait-restart', name)
        self.waited_for = 0
        self.respect_others = True
        self.taking_damage_counter = 0
        self.movement_see_range = G('movement-see-range', name)
        self.movement_path_range = G('movement-path-range', name)
        self.rotation_speed = G('rotation-speed', name)
        self.setAngle(random.randrange(0, 360))
        self.ai = G('ai', name)
        
    def updateActor(self, interval, world):
        """Update the enemy"""
        super(Enemy, self).updateActor(interval, world)
        #
        if not self.paused:
            #
            # Turn off the pain icon if we have recovered
            if self.taking_damage_counter:
                self.taking_damage_counter -= interval/1000.0
                if self.taking_damage_counter <= 0:
                    self.taking_damage_counter = 0
                    self.setSpriteName(self.name)
            #
            # Recover speed
            if self.speed < self.max_speed:
                self.speed = min(self.max_speed, self.speed + self.speed_recovery*interval/1000.0*self.max_speed/100.0)
            #
            # Create a goal and path if needed
            if not self.goal:
                self.findGoal()
            if not self.path:
                self.findPath()
            else:
                #
                # Try to move to the square but check with our parent to see
                # that the square is not occupied
                try:
                    self.move(*self.getMovementNormal(interval))
                except MovementBlocked:
                    #
                    # We are blocked - wait for a while in case this clears
                    self.waited_for += 1
                    if self.waited_for > self.wait_limit:
                        #
                        # Ok - not working out so ignore others and carry on moving
                        # This is a pragmatic approach to prevent deadlocks. Other 
                        # mechanisms should prevent this also but this is a worst case
                        # backup plan
                        self.respect_others = False
                        self.waited_for = self.wait_restart
                        self.log.debug('%s waited too long - not respecting others' % self.getNiceName())
            #
            # Reset the respecting others flag after a suitable time
            if not self.respect_others:
                self.waited_for -= 1
                if self.waited_for <= 0:
                    self.respect_others = True
                    self.waited_for = 0
                    self.log.debug('%s going back to respecting others' % self.getNiceName())

    def getMovementOrthogonal(self, interval):
        """Return the dx, dy movement for a single frame. Raise a MovementBlocked if we cannot move"""
        #
        # Move to path
        gx, gy = self.path[0]
        px, py = self.layout.getCoords((gx, gy))
        x, y = self.x, self.y
        if x == px and y == py:
            self.path.pop(0)
            if not self.path:
                self.arrivedAtGoal()
            return (0, 0)
        else:
            if self.parent.canGoToSquare((gx, gy), self) or not self.respect_others:
                dx = math.copysign(min(abs(px-x), interval/1000.0*self.speed), px-x)
                dy = math.copysign(min(abs(py-y), interval/1000.0*self.speed), py-y)
                return (dx, dy)
            else:
                raise MovementBlocked

    def getMovementNormal(self, interval):
        """Return the dx, dy movement for a single frame. Raise a MovementBlocked if we cannot move"""
        x, y = self.x, self.y
        cx, cy = self.layout.getLocation((x, y))
        #
        # Are we there yet?
        if self.board.getItemAt((cx, cy)).is_goal:
            self.arrivedAtGoal()
        #
        # Be repelled from all the walls
        position = common.Vec2d(x, y)
        f_repulsive = common.Vec2d(0, 0)

        for wx in range(cx-self.movement_see_range, cx+self.movement_see_range+1):
            for wy in range(cy-self.movement_see_range, cy+self.movement_see_range+1):
                try:
                    enemies = self.parent.getOccupiersAt((wx, wy))
                except board.OutOfRange:
                    pass
                else:
                    for enemy in enemies:
                        offset = common.Vec2d(enemy.x, enemy.y) - position                   
                        contribution = offset.normalized() * 2500.0/max(offset.length**2, 1.0)
                        f_repulsive = f_repulsive + contribution

        #
        # Check if we need to move to the next pathing point
        if len(self.path) >= 2:
            offset_1 = (common.Vec2d(self.layout.getCoords(self.path[0])) - position).length
            if offset_1 < 16:
                self.path.pop(0)
        
        #
        # Be attracted to pathing points
        f_attractive = common.Vec2d(0, 0)
        for idx, point in enumerate(self.path[:self.movement_path_range]):
            offset = common.Vec2d(self.layout.getCoords(point)) - position
            multiplier = 1 #if (self.parent.canGoToSquare(point, self) or not self.respect_others) else -0.5
            contribution = multiplier * offset.normalized() * 30.0 / (offset.length ** 0.05)
            f_attractive = f_attractive + contribution 
        #
        f = f_attractive - f_repulsive
        dx, dy = f * interval/1000.0 * self.speed / 50.0
        #
        self.setAngle(self.getAngle()+self.rotation_speed)
        return dx, dy
        
            
            
    def findGoal(self):
        """Find a suitable goal"""
        self.goal = self.board.getGoalPosition(self.goal_index)
        self.log.info('New goal %s for %s' % (self.goal, self.getNiceName()))

    def setGoalIndex(self, index):
        """Set the goal index"""
        self.goal_index = index
        self.goal = None
        self.path = None
        
    def findPath(self):
        """Find a path to our goal"""
        try:
            the_path = self.board.getBestPath(self.layout.getLocation((self.x, self.y)), self.goal, self.ai)
        except common.NoPath, err:
            self.log.debug('No path: %s' % err)
            the_path = None
        if the_path:
            self.path = the_path[1:]
            self.log.info('New path (length %d) for %s' % (len(self.path), self.getNiceName()))
            if not self.path:
                self.arrivedAtGoal()
        else:
            self.path = []
    
    def clearPath(self):
        """Clear the path to the goal"""
        self.path = None
        
    def arrivedAtGoal(self):
        """We have arrived"""
        self.log.info('%s has arrived' % self.getNiceName())
        self.broadcaster.processEvent((common.E_ENEMY_AT_GOAL, self))
        self.parent.enemyHitGoal(self)
        
    def takeDamage(self, bullet):
        """Hit by something - return True if this destroys the enemy"""
        self.hp -= bullet.damage
        self.speed = max(0.0, self.speed - bullet.slowdown/100.0*self.max_speed)
        if self.hp <= 0:
            self.world.scheduleActorRemoval(self)
            return True
        self.taking_damage_counter = G('enemy-damage-counter')
        self.setSpriteName(self.name + '-pain')
        return False
